<?php

declare(strict_types=1);

namespace Dotclear\Plugin\enhancePostContent;

use ArrayObject;
use Dotclear\App;
use Dotclear\Core\Backend\Favorites;
use Dotclear\Helper\Process\TraitProcess;
use Dotclear\Helper\Html\Form\Checkbox;
use Dotclear\Helper\Html\Form\Div;
use Dotclear\Helper\Html\Form\Fieldset;
use Dotclear\Helper\Html\Form\Img;
use Dotclear\Helper\Html\Form\Input;
use Dotclear\Helper\Html\Form\Label;
use Dotclear\Helper\Html\Form\Legend;
use Dotclear\Helper\Html\Form\Link;
use Dotclear\Helper\Html\Form\Note;
use Dotclear\Helper\Html\Form\Para;
use Dotclear\Helper\Html\Form\Text;
use Dotclear\Interface\Core\BlogSettingsInterface;

/**
 * @brief       enhancePostContent backend class.
 * @ingroup     enhancePostContent
 *
 * @author      Jean-Christian Denis
 * @copyright   GPL-2.0 https://www.gnu.org/licenses/gpl-2.0.html
 */
class Backend
{
    use TraitProcess;

    public static function init(): bool
    {
        return self::status(My::checkContext(My::BACKEND));
    }

    public static function process(): bool
    {
        if (!self::status()) {
            return false;
        }

        My::addBackendMenuItem();

        App::behavior()->addBehaviors([
            // backend user dashboard favorites icon
            'adminDashboardFavoritesV2' => function (Favorites $favs): void {
                $favs->register(My::id(), [
                    'title'       => My::name(),
                    'url'         => My::manageUrl(),
                    'small-icon'  => My::icons(),
                    'large-icon'  => My::icons(),
                    'permissions' => App::auth()->makePermissions([App::auth()::PERMISSION_CONTENT_ADMIN]),
                ]);
            },
            // backend user preference form
            'adminBlogPreferencesFormV2' => function (BlogSettingsInterface $blog_settings): void {
                $active           = (bool) $blog_settings->get(My::id())->get('active');
                $allowedtplvalues = Epc::blogAllowedTemplateValue();
                $allowedpubpages  = Epc::blogAllowedTemplatePage();

                echo
                (new Fieldset(My::id() . '_params'))
                    ->legend(new Legend((new Img(My::icons()[0]))->class('icon-small')->render() . ' ' . My::name()))
                    ->items([
                        (new Div())
                            ->class('two-cols')
                            ->items([
                                (new Div())
                                    ->class('col')
                                    ->items([
                                        // active
                                        (new Para())
                                            ->items([
                                                (new Checkbox('epc_active', $active))
                                                    ->value(1),
                                                (new Label(__('Enable plugin'), Label::OUTSIDE_LABEL_AFTER))
                                                    ->for('epc_active')
                                                    ->class('classic'),
                                            ]),
                                        (new Note())
                                            ->class('form-note')
                                            ->text(__('This enable public widgets and contents filter.')),
                                        (new Para())
                                            ->items([
                                                (new Link())
                                                    ->href(My::manageUrl())
                                                    ->text(__('Set content filters')),
                                            ]),
                                    ]),
                                (new Div())
                                    ->class('col')
                                    ->items([
                                        (new Text('h5', __('Extra'))),
                                        (new Text('p', __('This is a special feature to edit list of allowed template values and public pages where this plugin works.'))),
                                        // allowedtplvalues
                                        (new Para())->items([
                                            (new Label(__('Allowed DC template values:'), Label::OUTSIDE_LABEL_BEFORE))->for('epc_allowedtplvalues'),
                                            (new Input('epc_allowedtplvalues'))->size(100)->maxlength(0)->value(Epc::encodeMulti($allowedtplvalues)),
                                        ]),
                                        (new Note())
                                            ->class('form-note')
                                            ->text(__('Use "readable_name1:template_value1;readable_name2:template_value2;" like "entry content:EntryContent;entry excerpt:EntryExcerpt;".')),
                                        // allowedpubpages
                                        (new Para())->items([
                                            (new Label(__('Allowed public pages:'), Label::OUTSIDE_LABEL_BEFORE))->for('epc_allowedpubpages'),
                                            (new Input('epc_allowedpubpages'))->size(100)->maxlength(0)->value(Epc::encodeMulti($allowedpubpages)),
                                        ]),
                                        (new Note())
                                            ->class('form-note')
                                            ->text(__('Use "readable_name1:template_page1;readable_name2:template_page2;" like "post page:post.html;home page:home.html;".')),
                                    ]),
                            ]),
                        (new Text('br'))
                            ->class('clear'),
                    ])
                    ->render();
            },
            // backend user preference save
            'adminBeforeBlogSettingsUpdate' => function (BlogSettingsInterface $blog_settings): void {
                $active           = !empty($_POST['epc_active']);
                $allowedtplvalues = Epc::decodeMulti($_POST['epc_allowedtplvalues']);
                $allowedpubpages  = Epc::decodeMulti($_POST['epc_allowedpubpages']);

                $blog_settings->get(My::id())->put('active', $active);
                $blog_settings->get(My::id())->put('allowedtplvalues', json_encode($allowedtplvalues));
                $blog_settings->get(My::id())->put('allowedpubpages', json_encode($allowedpubpages));
            },
            // backend epc list filter
            'adminFiltersListsV2' => function (ArrayObject $sorts): void {
                $sorts['epc'] = [
                    My::name(),
                    [
                        __('Date')  => 'epc_upddt',
                        __('Key')   => 'epc_key',
                        __('Value') => 'epc_value',
                        __('ID')    => 'epc_id',
                    ],
                    'epc_upddt',
                    'desc',
                    [__('records per page'), 20],
                ];
            },
            // widgets registration
            'initWidgets' => Widgets::initWidgets(...),
        ]);

        return true;
    }
}
